<?php
require __DIR__ . '/includes/config.php';
require __DIR__ . '/includes/functions.php';

//
// Small helper: handle file upload and return relative path like "uploads/banners/file.jpg"
//

function handle_upload(string $field, string $subdir, array $allowedExts, int $maxSizeMB, array &$errors, bool $required = true)
{
    if (!isset($_FILES[$field]) || $_FILES[$field]['error'] === UPLOAD_ERR_NO_FILE) {
        if ($required) {
            $errors[] = "Please upload a file for " . htmlspecialchars($field) . ".";
        }
        return null;
    }

    $file = $_FILES[$field];

    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errors[] = "There was a problem uploading " . htmlspecialchars($field) . ".";
        return null;
    }

    if ($file['size'] > $maxSizeMB * 1024 * 1024) {
        $errors[] = ucfirst($field) . " is too large. Max size is {$maxSizeMB}MB.";
        return null;
    }

    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, $allowedExts, true)) {
        $errors[] = ucfirst($field) . " must be one of: " . implode(', ', $allowedExts) . ".";
        return null;
    }

    $uploadBase = __DIR__ . '/uploads';
    $targetDir  = $uploadBase . '/' . $subdir;

    if (!is_dir($targetDir)) {
        @mkdir($targetDir, 0775, true);
    }

    $safeExt    = preg_replace('/[^a-z0-9]/i', '', $ext);
    $uniqueName = $field . '_' . uniqid('', true) . '.' . $safeExt;
    $targetPath = $targetDir . '/' . $uniqueName;

    if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
        $errors[] = "Unable to save upload for " . htmlspecialchars($field) . ".";
        return null;
    }

    // Return path relative to web root (assuming submit-artist.php is at web root)
    return 'uploads/' . $subdir . '/' . $uniqueName;
}

//
// PACKAGES / ADDONS DEFINITIONS
//
$availablePackages = [
    '14day' => [
        'code'   => '14day',
        'name'   => '14 Day Artist Submission',
        'amount' => 10.00,
        'label'  => '14 Days • £10.00'
    ],
    '30day' => [
        'code'   => '30day',
        'name'   => '30 Day Artist Submission',
        'amount' => 20.00,
        'label'  => '30 Days • £20.00'
    ],
    'label' => [
        'code'   => 'label',
        'name'   => 'Label Offer (Up to 50 Artists)',
        'amount' => 50.00,
        'label'  => 'Label Offer • £50.00 / month'
    ],
];

$availableAddons = [
    'social' => [
        'code'   => 'social',
        'label'  => 'Social Media Shoutouts',
        'amount' => 10.00
    ],
    'app' => [
        'code'   => 'app',
        'label'  => 'App Appearance / Featured',
        'amount' => 20.00
    ],
    'website' => [
        'code'   => 'website',
        'label'  => 'Custom Artist Website Build',
        'amount' => 100.00
    ],
];

$errors       = [];
$success      = false;
$createdEmail = '';
$invoiceData  = null;

// these are used in the success view
$name  = '';
$email = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name           = trim($_POST['name'] ?? '');
    $email          = trim($_POST['email'] ?? '');
    $password       = $_POST['password'] ?? '';
    $tagline        = trim($_POST['tagline'] ?? '');
    $bio            = trim($_POST['bio'] ?? '');
    $instagram      = trim($_POST['instagram'] ?? '');
    $youtube        = trim($_POST['youtube'] ?? '');

    $track_title    = trim($_POST['track_title'] ?? '');
    $track_subtitle = trim($_POST['track_subtitle'] ?? '');
    $package_code   = $_POST['package_code'] ?? '';

    $addon_social   = !empty($_POST['addon_social']);
    $addon_app      = !empty($_POST['addon_app']);
    $addon_website  = !empty($_POST['addon_website']);

    // --- basic validation (text inputs) -------------------------
    if ($name === '') {
        $errors[] = "Artist / stage name is required.";
    }
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "A valid email address is required.";
    }
    if (strlen($password) < 6) {
        $errors[] = "Password must be at least 6 characters.";
    }
    if ($track_title === '') {
        $errors[] = "Track title is required.";
    }
    if (!isset($availablePackages[$package_code])) {
        $errors[] = "Please choose a promotion package.";
    }

    // check email uniqueness
    if (!$errors) {
        $users = read_json('users.json');
        foreach ($users as $u) {
            if (strtolower($u['email']) === strtolower($email)) {
                $errors[] = "This email is already registered. Please login instead.";
                break;
            }
        }
    }

    // --- handle uploads -----------------------------------------
    $hero_banner   = null;
    $profile_image = null;
    $track_audio   = null;
    $track_artwork = null;

    if (!$errors) {
        // banner image (required)
        $hero_banner = handle_upload(
            'hero_banner',
            'banners',
            ['jpg', 'jpeg', 'png', 'webp'],
            10,  // MB
            $errors,
            true
        );

        // profile image (required)
        $profile_image = handle_upload(
            'profile_image',
            'profiles',
            ['jpg', 'jpeg', 'png', 'webp'],
            10,
            $errors,
            true
        );

        // audio (required)
        $track_audio = handle_upload(
            'track_audio',
            'audio',
            ['mp3', 'wav', 'ogg', 'm4a'],
            40,
            $errors,
            true
        );

        // artwork (optional)
        $track_artwork = handle_upload(
            'track_artwork',
            'artwork',
            ['jpg', 'jpeg', 'png', 'webp'],
            10,
            $errors,
            false
        );

        if (!$hero_banner || !$profile_image || !$track_audio) {
            // handle_upload has already pushed error messages
        }
    }

    // --- create user, artist, invoice if still no errors --------
    if (!$errors) {
        $users    = read_json('users.json');
        $artists  = read_json('artists.json');
        $invoices = read_json('invoices.json');

        // USER
        $newUserId = $users ? (max(array_column($users, 'id')) + 1) : 1;
        $slug      = slugify($name) . '-' . $newUserId;

        $users[] = [
            "id"            => $newUserId,
            "role"          => "artist",
            "name"          => $name,
            "email"         => $email,
            "password_hash" => password_hash($password, PASSWORD_DEFAULT),
            "artist_slug"   => $slug,
            "status"        => "pending"
        ];
        write_json('users.json', $users);

        // ARTIST
        $bioParagraphs = [];
        if ($bio !== '') {
            $bioParagraphs = array_values(
                array_filter(array_map('trim', preg_split('/\r\n|\r|\n/', $bio)))
            );
        }

        $newArtistId = $artists ? (max(array_column($artists, 'id')) + 1) : 1;

        $artistEntry = [
            "id"            => $newArtistId,
            "slug"          => $slug,
            "name"          => $name,
            "tagline"       => $tagline ?: "New submission on Londons Energy Radio",
            "quote"         => "",
            "quote_credit"  => "",
            "hero_banner"   => $hero_banner,
            "profile_image" => $profile_image,
            "bio"           => $bioParagraphs,
            "socials"       => [
                "instagram" => $instagram,
                "youtube"   => $youtube
            ],
            "tracks" => [
                [
                    "title"       => $track_title,
                    "subtitle"    => $track_subtitle,
                    "artist_name" => $name,
                    "audio_url"   => $track_audio,
                    "artwork"     => $track_artwork ?: $profile_image,
                    "is_featured" => true
                ]
            ],
            "featured_on_app" => false,
            // submission status handled by admin:
            // pending / on_hold / approved / featured / expired / cancelled
            "status"          => "pending",
            "owner_user_id"   => $newUserId
        ];

        $artists[] = $artistEntry;
        write_json('artists.json', $artists);

        // INVOICE
        $selectedPackage = $availablePackages[$package_code];
        $totalAmount     = $selectedPackage['amount'];
        $addonsSelected  = [];

        if ($addon_social) {
            $totalAmount += $availableAddons['social']['amount'];
            $addonsSelected[] = $availableAddons['social']['label'];
        }
        if ($addon_app) {
            $totalAmount += $availableAddons['app']['amount'];
            $addonsSelected[] = $availableAddons['app']['label'];
        }
        if ($addon_website) {
            $totalAmount += $availableAddons['website']['amount'];
            $addonsSelected[] = $availableAddons['website']['label'];
        }

        $newInvoiceId = $invoices ? (max(array_column($invoices, 'id')) + 1) : 1;

        $invoiceEntry = [
            "id"            => $newInvoiceId,
            "user_id"       => $newUserId,
            "artist_id"     => $newArtistId,
            "package_code"  => $selectedPackage['code'],
            "package_name"  => $selectedPackage['name'],
            "amount"        => $totalAmount,
            "currency"      => "GBP",
            // invoice status (admin dropdown):
            // pending / paid / cancelled / refunded
            "status"        => "pending",
            "addons"        => $addonsSelected,
            "payment_url"   => "https://revolut.me/ck_09",
            "created_at"    => date('Y-m-d H:i:s'),
            "updated_at"    => date('Y-m-d H:i:s')
        ];

        $invoices[] = $invoiceEntry;
        write_json('invoices.json', $invoices);

        $success      = true;
        $createdEmail = $email;
        $invoiceData  = $invoiceEntry;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Submit Your Music • Londons Energy Radio</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <link rel="stylesheet" href="assets/css/main.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
  <script src="assets/js/main.js" defer></script>
</head>
<body class="dark-bg">
  <div class="page-bg"></div>

  <!-- TOP NAV -->
  <header class="site-header site-header--sticky">
    <div class="site-header-inner">
      <a href="index.php" class="site-brand">
        <img
          src="https://londonsenergyradio.co.uk/wp-content/uploads/2022/12/Energy-radio-p3h439ea3kdydq5pmbj4z4m1ek89863lffdvjk6qmc.png"
          alt="Londons Energy Radio"
          class="site-logo"
        >
        <div class="site-brand-text">
          <span class="site-brand-top">Londons Energy Radio</span>
          <span class="site-brand-sub">Artist Portal</span>
        </div>
      </a>

      <button class="nav-toggle" type="button" aria-controls="primary-nav" aria-expanded="false">

        <span class="sr-only">Menu</span>

        <i class="fa-solid fa-bars"></i>

      </button>


      <nav class="site-nav" id="primary-nav" aria-label="Primary">
        <a href="index.php" class="nav-link">Artists</a>
        <a href="packages.php" class="nav-link">Packages</a>
        <a href="submit-artist.php" class="nav-link nav-link--active">Submit Music</a>
        <a href="login.php" class="nav-link nav-link--outline">
          <i class="fa-solid fa-user"></i>
          Login
        </a>
      </nav>
    </div>
  </header>

  <div class="nav-overlay" data-nav-close></div>

  <main>
    <!-- HERO -->
    <section class="packages-hero">
      <div class="shell">
        <div class="packages-hero-inner">
          <div>
            <p class="hero-kicker">Artist Submission</p>
            <h1>Submit Your Artist Profile</h1>
            <p class="hero-copy">
              Upload your banner, profile photo and featured track directly from your device.
              We’ll create your portal login, artist profile and invoice. Once payment is confirmed
              and your submission is approved, your profile goes live on Londons Energy Radio.
            </p>
          </div>
          <div class="packages-hero-tag">
            <span><i class="fa-solid fa-id-card"></i> Login with your email & password</span>
            <span><i class="fa-solid fa-credit-card"></i> Invoice created & pay via Revolut</span>
            <span><i class="fa-solid fa-gauge-high"></i> Track submission & invoice status in your dashboard</span>
          </div>
        </div>
      </div>
    </section>

    <section class="shell" style="padding-bottom:3rem;">

      <?php if ($success && $invoiceData): ?>
        <div class="dash-card">
          <h2>Submission received</h2>
          <p>
            Thank you, <strong><?= htmlspecialchars($name) ?></strong>.  
            Your artist account, profile and invoice have been created and are currently
            <strong>pending review</strong>.
          </p>

          <p style="margin-top:0.6rem;">
            You can now log in any time using:
          </p>
          <ul style="font-size:0.85rem;color:var(--text-muted);margin-top:0.3rem;">
            <li><strong>Email:</strong> <?= htmlspecialchars($createdEmail) ?></li>
            <li><strong>Password:</strong> the password you just set</li>
          </ul>
          <p style="font-size:0.85rem;color:var(--text-muted);margin-top:0.4rem;">
            Inside your dashboard you’ll be able to
            <strong>track and edit your submission</strong>, manage additional tracks
            and view invoices.  
            Submission status flows through: <em>Pending → On Hold → Approved → Featured / Expired / Cancelled</em>.
          </p>

          <hr style="border:none;border-top:1px solid rgba(255,255,255,0.1);margin:1rem 0;">

          <h3 style="font-size:0.9rem;margin:0 0 0.4rem;">Invoice created</h3>
          <ul style="font-size:0.85rem;color:var(--text-muted);margin-top:0.2rem;">
            <li><strong>Invoice #</strong> <?= (int)$invoiceData['id'] ?></li>
            <li><strong>Package:</strong> <?= htmlspecialchars($invoiceData['package_name']) ?></li>
            <li><strong>Amount:</strong> £<?= number_format($invoiceData['amount'], 2) ?></li>
            <?php if (!empty($invoiceData['addons'])): ?>
              <li><strong>Add-ons:</strong> <?= htmlspecialchars(implode(', ', $invoiceData['addons'])) ?></li>
            <?php endif; ?>
            <li><strong>Status:</strong> Pending (admin will mark as Paid / Cancelled / Refunded)</li>
          </ul>

          <p style="font-size:0.8rem;color:var(--text-muted);margin-top:0.6rem;">
            Complete your payment via Revolut using the button below. After paying, return to the portal
            and log in to see your invoice and submission. Admin will set the invoice status to
            <strong>PAID</strong> once confirmed.
          </p>

          <div style="margin-top:1rem;display:flex;flex-wrap:wrap;gap:0.6rem;">
            <a href="<?= htmlspecialchars($invoiceData['payment_url']) ?>" target="_blank" class="btn-primary">
              <i class="fa-solid fa-credit-card"></i>
              Pay with Revolut
            </a>
            <a href="login.php" class="btn-secondary">
              <i class="fa-solid fa-user"></i>
              Go to Login
            </a>
            <a href="index.php" class="btn-tertiary">
              ← Back to Portal
            </a>
          </div>
        </div>

        <!-- Auto-open Revolut in new tab -->
        <script>
          window.addEventListener('load', function () {
            try {
              window.open('<?= htmlspecialchars($invoiceData['payment_url']) ?>', '_blank');
            } catch (e) {}
          });
        </script>

      <?php else: ?>

        <?php if ($errors): ?>
          <div class="dash-card" style="border-color:#ff4b4b;">
            <h2>Check your details</h2>
            <ul>
              <?php foreach ($errors as $e): ?>
                <li><?= htmlspecialchars($e) ?></li>
              <?php endforeach; ?>
            </ul>
            <p style="font-size:0.75rem;color:var(--text-muted);margin-top:0.4rem;">
              Note: for security reasons you’ll need to re-select your files when correcting the form.
            </p>
          </div>
        <?php endif; ?>

        <form method="post" class="dash-card" enctype="multipart/form-data">
          <h2>Artist & Account Details</h2>

          <p style="font-size:0.8rem;color:var(--text-muted);margin-top:0;margin-bottom:0.9rem;">
            Step 1: Fill in your details and upload your files.  
            Step 2: We create your login, submission and invoice.  
            Step 3: You pay via Revolut and then log in to track everything.
          </p>

          <label>Artist / Stage Name *</label>
          <input type="text" name="name" required
                 value="<?= htmlspecialchars($_POST['name'] ?? '') ?>">

          <label>Email (for login & contact) *</label>
          <input type="email" name="email" required
                 value="<?= htmlspecialchars($_POST['email'] ?? '') ?>">

          <label>Password *</label>
          <input type="password" name="password" required>

          <label>Tagline (genre / style)</label>
          <input type="text" name="tagline"
                 placeholder="Hip-Hop • Authentic • Lyrical Depth"
                 value="<?= htmlspecialchars($_POST['tagline'] ?? '') ?>">

          <label>Short Bio (each new line becomes a paragraph)</label>
          <textarea name="bio" rows="5"><?= htmlspecialchars($_POST['bio'] ?? '') ?></textarea>

          <h2 style="margin-top:1.5rem;">Images & Media Upload</h2>

          <label>Banner Image (cover at top of profile) *</label>
          <input type="file" name="hero_banner" accept="image/*" required>

          <label>Profile Image (square/portrait) *</label>
          <input type="file" name="profile_image" accept="image/*" required>

          <label>Instagram URL</label>
          <input type="url" name="instagram"
                 placeholder="https://www.instagram.com/yourhandle"
                 value="<?= htmlspecialchars($_POST['instagram'] ?? '') ?>">

          <label>YouTube URL</label>
          <input type="url" name="youtube"
                 placeholder="https://youtube.com/@yourchannel"
                 value="<?= htmlspecialchars($_POST['youtube'] ?? '') ?>">

          <h2 style="margin-top:1.5rem;">Promo Package & Add-ons</h2>

          <label>Choose your package *</label>
          <select name="package_code" required>
            <option value="">Select a package…</option>
            <?php foreach ($availablePackages as $pkg): ?>
              <option value="<?= htmlspecialchars($pkg['code']) ?>"
                <?= (($_POST['package_code'] ?? '') === $pkg['code']) ? 'selected' : '' ?>>
                <?= htmlspecialchars($pkg['label']) ?>
              </option>
            <?php endforeach; ?>
          </select>

          <div style="margin-top:0.6rem;font-size:0.8rem;color:var(--text-muted);">
            Optional add-ons (added to your invoice total):
          </div>

          <label style="display:flex;align-items:center;gap:0.45rem;margin-top:0.4rem;font-size:0.8rem;">
            <input type="checkbox" name="addon_social" value="1"
              <?= !empty($_POST['addon_social']) ? 'checked' : '' ?>>
            + £10 Social Media Shoutouts
          </label>

          <label style="display:flex;align-items:center;gap:0.45rem;margin-top:0.25rem;font-size:0.8rem;">
            <input type="checkbox" name="addon_app" value="1"
              <?= !empty($_POST['addon_app']) ? 'checked' : '' ?>>
            + £20 App Appearance / Featured
          </label>

          <label style="display:flex;align-items:center;gap:0.45rem;margin-top:0.25rem;font-size:0.8rem;">
            <input type="checkbox" name="addon_website" value="1"
              <?= !empty($_POST['addon_website']) ? 'checked' : '' ?>>
            + £100 Custom Artist Website Build
          </label>

          <h2 style="margin-top:1.5rem;">Featured Track</h2>

          <label>Track Title *</label>
          <input type="text" name="track_title" required
                 placeholder="Your single title"
                 value="<?= htmlspecialchars($_POST['track_title'] ?? '') ?>">

          <label>Track Subtitle (e.g. ft. artist)</label>
          <input type="text" name="track_subtitle"
                 placeholder="(Ft. Guest Artist)"
                 value="<?= htmlspecialchars($_POST['track_subtitle'] ?? '') ?>">

          <label>Track Audio File (MP3 / WAV / OGG / M4A) *</label>
          <input type="file" name="track_audio" accept=".mp3,.wav,.ogg,.m4a,audio/*" required>

          <label>Track Artwork (optional)</label>
          <input type="file" name="track_artwork" accept="image/*">

          <div style="margin-top:1.2rem;display:flex;flex-wrap:wrap;gap:0.6rem;">
            <button type="submit" class="btn-primary">
              <i class="fa-solid fa-paper-plane"></i>
              Submit & Create Invoice
            </button>
            <a href="index.php" class="btn-tertiary">← Back to Portal</a>
          </div>
        </form>
      <?php endif; ?>
    </section>
  </main>

  <footer class="site-footer">
    KEEPING IT ENERGY • LONDONS ENERGY RADIO
  </footer>
</body>
</html>
